-- Polygon Editing
-- by John Millard

-- This is a bonus because it's very advanced, and because it was originally meant
-- to be included with the regular PhysicsLab but wasn't ready in time.
-- Also, unlike the other tests, this doesn't showcase any new abilities of PhysicsLab.
-- In other words there is no 'PhysicsLab:makeEditablePolygon(...)' command.
-- if you want to use one in your own projects you'll have to go digging through the
-- code and figure out how it's done!

Bonus_EditablePolygon = class()

function Bonus_EditablePolygon:init()
    self.title = "BONUS: editable polygon"
end

function Bonus_EditablePolygon:setup()
    self.lab = PhysicsLab()
    
    print("Here's one that's just for fun. Tap in three places to set a starting shape. Drag points to move them and drag on any line to add new points.\n\nCircles will soon drop and bounce off whatever you make!")
    
    self.verts = {}
    self.index = -1
    self.touchID = -1
    self.timer = 0
    
    -- the mesh to draw the polygon with
    self.polyMesh = mesh()
    -- rigid body for the polygon
    self.polyBody = nil
    
end

function Bonus_EditablePolygon:draw()
    -- create a circle every 2 seconds
    self.timer = self.timer + DeltaTime
    if self.timer > 2 then
        local body = self.lab:circleAt(WIDTH/2, HEIGHT, 25)
        body.restitution = 0.5
        self.timer = 0
    end
    
    background(0, 0, 0)
    self.lab:draw()
    
    -- draw the polygon interia
    fill(95, 164, 183, 131)
    self.polyMesh:draw()
    pushStyle()
    lineCapMode(PROJECT)
    fill(255, 255, 255, 255)
    
    -- draw the polygon outline
    local pv = self.verts[1]
    for k,v in ipairs(self.verts) do
        noStroke()
        ellipse(v.x, v.y, 10, 10)
        stroke(79, 183, 233)
        line(pv.x, pv.y, v.x, v.y)
        pv = v
    end
    if pv then
        line(pv.x, pv.y, self.verts[1].x, self.verts[1].y)
    end
    popStyle()
    
end

function Bonus_EditablePolygon:touched(touch)
    local tv = vec2(touch.x, touch.y)
    
    if touch.state == BEGAN and self.index == -1 then        
        -- find the closest vertex within 50 px of thr touch
        self.touchID = touch.id
        local minDist = math.huge
        for k,v in ipairs(self.verts) do
            local dist = v:dist(tv)
            if dist < minDist and dist < 50 then
                minDist = dist
                self.index = k
            end
        end
        
        -- if no point is found near the touch, insert a new one           
        if self.index == -1 then
            self.index = #self.verts
            if self.index == 0 then
                self.index = self.index + 1
            end
            
            -- if touch is within 50px to a line, insert point on line
            if #self.verts > 2 then
                local minDist = math.huge
                local pv = self.verts[self.index]
                for k,v in ipairs(self.verts) do
                    local dist = self:distPointToLineSeg(tv, pv, v)
                    if dist < minDist and dist < 50 then
                        minDist = dist
                        self.index = k
                    end
                    pv = v
                end
            end
            
            table.insert(self.verts, self.index, tv)
        else
            self.verts[self.index] = tv
        end
        
    elseif touch.state == MOVING and touch.id == self.touchID then
        self.verts[self.index] = tv 
    elseif touch.state == ENDED and touch.id == self.touchID then
        self.index = -1
    end
    --    if #polyMesh.vertices > 3 then mapMeshCoords(polyMesh) end
    -- use triangulate to generate triangles from the polygon outline for the mesh
    self.polyMesh.vertices = triangulate(self.verts)
    
    
    if self.polyBody then
        self.lab:destroy(self.polyBody)
    end
    if #self.verts > 2 then
        self.polyBody = self.lab:polygonAt(0, 0, {table.unpack(self.verts)})
        self.polyBody.type = STATIC
    end
end

-- This function iterates through the mesh
--  and returns two values defining the bounds
--  returns lower left (vec3), upper right (vec3)

function Bonus_EditablePolygon:meshBounds( m )
    local verts = m.vertices
    
    local ll = vec3( math.huge, math.huge, math.huge )
    local ur = -ll
    local v
    
    for i = 1,#verts do
        v = verts[i]
        
        if v.x < ll.x then ll.x = v.x end
        if v.y < ll.y then ll.y = v.y end
        if v.z < ll.z then ll.z = v.z end
        if v.x > ur.x then ur.x = v.x end
        if v.y > ur.y then ur.y = v.y end
        if v.z > ur.z then ur.z = v.z end
    end
    
    return ll, ur
end

function Bonus_EditablePolygon:mapMeshCoords( m )
    
    for k, v in pairs(m.vertices) do
        print(k, " ", v)
    end
    
    local ll, ur = self:meshBounds( m )
    
    local bounds = vec2( ur.x - ll.x, ur.y - ll.y )
    
    local v, tx, ty
    for i = 1, m.size do
        
        v = m:vertex(i)
        print(v, i)
        tx = (v.x - ll.x) / bounds.x
        ty = (v.y - ll.y) / bounds.y        
        print(tx, ty)
        print(m:texCoord( i))
        m:texCoord( i , tx, ty ) 
        
    end
end

-- distPointToLineSeg(): shortest distance of a point to a line segment.
function Bonus_EditablePolygon:distPointToLineSeg(p , s1, s2)
    local v = s2 - s1
    local w = p - s1
    
    c1 = w:dot(v)
    if c1 <= 0 then
        return p:dist(s1)
    end
    
    c2 = v:dot(v)
    if c2 <= c1 then
        return p:dist(s2)
    end
    
    b = c1 / c2;
    pb = s1 + b * v;
    return p:dist(pb)
end

function Bonus_EditablePolygon:collide(contact)
    self.lab:collide(contact)
end

function Bonus_EditablePolygon:cleanup()
    self.lab:clear()
end

